/****************************************************************************
 *
 *                                 M U E S L I   v 1.4
 *
 *
 *     Copyright 2016 IMDEA Materials Institute, Getafe, Madrid, Spain
 *     Contact: muesli.materials@imdea.org
 *     Author: Ignacio Romero (ignacio.romero@imdea.org)
 *
 *     This file is part of MUESLI.
 *
 *     MUESLI is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     MUESLI is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with MUESLI.  If not, see <http://www.gnu.org/licenses/>.
 *
*****************************************************************************/



#include "reducedfinitestrain.h"

/*
 *
 * reducedsmallstrain.cpp
 * reduced models with zero stresses, such as shell and beams
 * D. Portillo, may 2017
 *
 * see "Using finite strain 3D-material models in beam and 
 *      shell elements", S. Klinkel & S. Govindjee
 *
 * Note that since C = 2E + I is positive semidefinite, the problem 
 * in finite strain theory might not have solution.
 */

const int maxiter = 50;
const double tol = 1.0e-10;
const double coefE[6] ={1.0, 1.0, 1.0, 2.0, 2.0, 2.0};

using namespace muesli;


rFiniteStrainMP :: rFiniteStrainMP(finiteStrainMP *mp) 
{
    theFiniteStrainMP = const_cast<finiteStrainMP*>(mp);
}





void rFiniteStrainMP :: CauchyStressVector(double S[6]) const
{
    istensor sigma;
    theFiniteStrainMP -> CauchyStress(sigma);
    muesli::tensorToVector(sigma, S);
}





void rFiniteStrainMP :: commitCurrentState()
{
    theFiniteStrainMP -> commitCurrentState();
}




itensor& rFiniteStrainMP :: deformationGradient()
{
    return theFiniteStrainMP -> deformationGradient();
}




const itensor& rFiniteStrainMP :: deformationGradient() const
{
    return theFiniteStrainMP -> deformationGradient();
}




double rFiniteStrainMP :: effectiveStoredEnergy() const
{
    return theFiniteStrainMP -> effectiveStoredEnergy();
}




void rFiniteStrainMP :: energyMomentumTensor(const itensor &F, itensor &T) const
{
    theFiniteStrainMP -> energyMomentumTensor(F, T);
}




void rFiniteStrainMP :: firstPiolaKirchhoffStress(itensor &P) const
{
    theFiniteStrainMP -> firstPiolaKirchhoffStress(P);
}




materialState rFiniteStrainMP :: getConvergedState() const
{
    return theFiniteStrainMP -> getConvergedState();
}




materialState rFiniteStrainMP :: getCurrentState() const
{
    return theFiniteStrainMP -> getCurrentState();
}




void rFiniteStrainMP :: KirchhoffStress(istensor &tau) const
{
    theFiniteStrainMP -> KirchhoffStress(tau);
}





void rFiniteStrainMP :: KirchhoffStressVector(double tauv[6]) const
{
    theFiniteStrainMP -> KirchhoffStressVector(tauv);
}




void rFiniteStrainMP :: resetCurrentState()
{
    theFiniteStrainMP -> resetCurrentState();
}




void rFiniteStrainMP :: secondPiolaKirchhoffStress(istensor& S) const
{
    theFiniteStrainMP -> secondPiolaKirchhoffStress(S);
}




void  rFiniteStrainMP :: setRandom()
{
    theFiniteStrainMP -> setRandom();
}




double rFiniteStrainMP :: volumetricStiffness() const
{
    return theFiniteStrainMP -> volumetricStiffness();
}




bool rFiniteStrainMP :: testImplementation(std::ostream& of, const bool testDE, const bool testDDE) const
{
    
    bool isok = true;
    // set a random update in the material
    itensor F;
    F.setRandom();
    F *= 1.0e-10;
    F = itensor::identity();
    if (F.determinant() < 0.0) F *= -1.0;
    rFiniteStrainMP& theMP = const_cast<rFiniteStrainMP&>(*this);
    theMP.updateCurrentState(0.0, F);
    theMP.commitCurrentState();

    double tn1 = muesli::randomUniform(0.1,1.0);
    istensor u;
    u.setRandom();
    u *= 1e-2;
    F += u;
    istensor FtF;
    FtF = istensor::tensorTransposedTimesTensor(F);
    istensor Uc;
    Uc = istensor::squareRoot(FtF);
    itensor R;
    R = F*Uc.inverse();
    istensor Ec;
    Ec = 0.5 * ( istensor::tensorTransposedTimesTensor(F) - istensor::identity() );

    if (F.determinant() < 0.0) F *= -1.0;
    theMP.updateCurrentState(tn1, F);


    istensor S;
    secondPiolaKirchhoffStress(S);

    itensor4 nC;
    istensor dS, Sp1, Sp2, Sm1, Sm2;


    // numerical differentiation stress
    istensor numS;
    numS.setZero();
    
    const double inc = 1.0e-3*Ec.norm();
    for (unsigned i=0; i<3; i++)
    {
        for (unsigned j=i; j<3; j++)
        {
            double original = Ec(i,j);

            Ec(i,j) = Ec(j,i) =  original + inc;
            FtF = 2.0*Ec + istensor::identity();
            Uc = istensor::squareRoot(FtF);
            F = R*Uc;
            theMP.updateCurrentState(tn1, F);
            double Wp1 = effectiveStoredEnergy();
            secondPiolaKirchhoffStress(Sp1);

            Ec(i,j) = Ec(j,i) = original + 2.0*inc;
            FtF = 2.0*Ec + istensor::identity();
            Uc = istensor::squareRoot(FtF);
            F = R*Uc;
            theMP.updateCurrentState(tn1, F);
            double Wp2 = effectiveStoredEnergy();
            secondPiolaKirchhoffStress(Sp2);

            Ec(i,j) = Ec(j,i) = original - inc;
            FtF = 2.0*Ec + istensor::identity();
            Uc = istensor::squareRoot(FtF);
            F = R*Uc;
            theMP.updateCurrentState(tn1, F);
            double Wm1 = effectiveStoredEnergy();
            secondPiolaKirchhoffStress(Sm1);

            Ec(i,j) = Ec(j,i) = original - 2.0*inc;
            FtF = 2.0*Ec + istensor::identity();
            Uc = istensor::squareRoot(FtF);
            F = R*Uc;
            theMP.updateCurrentState(tn1, F);
            double Wm2 = effectiveStoredEnergy();
            secondPiolaKirchhoffStress(Sm2);

            // fourth order approximation of the derivative
            numS(i,j) = (-Wp2 + 8.0*Wp1 - 8.0*Wm1 + Wm2)/(12.0*inc);
	    if (i!=j) numS(i,j) *= 0.5;
	    numS(j,i) = numS(i,j);

            // derivative of PK stress
            dS = (-Sp2 + 8.0*Sp1 - 8.0*Sm1 + Sm2)/(12.0*inc);
	    if (i!=j) dS *= 0.5;
            for (unsigned k=0; k<3; k++)
                for (unsigned l=0; l<3; l++)
                    nC(k,l,i,j) = nC(k,l,j,i) = dS(k,l);

            Ec(i,j) = Ec(j,i) = original;
            FtF = 2.0*Ec + istensor::identity();
            Uc = istensor::squareRoot(FtF);
            F = R*Uc;
            theMP.updateCurrentState(tn1, F);
        }
    }
    // compare DEnergy with the derivative of Energy
    {
        if (testDE)
        {
            istensor errorS = numS - S;
            isok = (errorS.norm()/S.norm() < 1e-4);
            of << "\n   1. Comparing S with derivative of Weff.";
            if (isok)
            {
                of << " Test passed.";
            }
            else
            {
                of << "\n Relative error in DE computation: " << errorS.norm()/S.norm() << ". Test failed.";
                of << "\n S: \n" << S;
                of << "\n numS: \n" << numS;
            }
        }
        else
        {
            of << "\n   1. Comparing S with derivative of Weff ::: not run for this material";
        }
        of << std::flush;
    }

    // (2) compare tensor c with derivative of stress reduced formulation
    if ((true))
    {
	// get the voigt notation
	double nCv[6][6];
	muesli::tensorToMatrix(nC,nCv);
		
	// get the tangent in voigt notation
	double tgv[6][6];
	convectedTangentMatrix(tgv);

        // relative error less than 0.01%
        double error = 0.0;
        double norm = 0.0;
        for (unsigned i=0; i<6; i++)
            for (unsigned j=0; j<6; j++)
            {
            	error += pow(nCv[i][j]-tgv[i][j],2);
            	norm  += pow(tgv[i][j],2);
            }
        error = sqrt(error);
        norm = sqrt(norm);
        isok = (error/norm < 1e-4);

        of << "\n   2. Comparing tensor C with DStress (reduced model).";
        if (isok)
        {
            of << " Test passed.";
        }
        else
        {
            of << "\n      Test failed.";
            of << "\n      Relative error in DWeff computation: " <<  error/norm;
        }
    }


    return isok;
}




double rFiniteStrainMP :: plasticSlip() const
{
    return theFiniteStrainMP -> plasticSlip();
}




void rFiniteStrainMP :: CauchyStress(istensor &sigma) const
{
    return theFiniteStrainMP -> CauchyStress(sigma);
}




double rFiniteStrainMP :: storedEnergy() const
{
    return theFiniteStrainMP -> storedEnergy();
}




double rFiniteStrainMP :: waveVelocity() const
{
    return theFiniteStrainMP -> waveVelocity();
}




double rFiniteStrainMP :: dissipatedEnergy() const
{
    return theFiniteStrainMP -> dissipatedEnergy();
}




/*********************************************************************************
  reduced1zFMP
**********************************************************************************/


reduced1zFMP :: reduced1zFMP(finiteStrainMP *mp, int inmapim[5], int inmapiz[1]) :
muesli::rFiniteStrainMP(mp)
{
    for (int i=0; i<5; i++)
        mapim[i] = inmapim[i];

    mapiz[0] = inmapiz[0];
} 




reduced1zFMP :: ~reduced1zFMP()
{

}




void reduced1zFMP :: convectedTangentMatrix(double C[6][6]) const
{
    itensor4 tg;
    theFiniteStrainMP -> convectedTangent(tg);
    muesli::tensorToMatrix(tg, C);

    int iz = mapiz[0]; // Cc's index such that Cc[iz][iz] = Czz
    double Czz = C[iz][iz];
    double Cmz[5] = {0.0};
    double Czm[5] = {0.0};

    for (int i=0; i<5; i++)
    {
       Cmz[i] = C[mapim[i]][iz      ];
       Czm[i] = C[iz      ][mapim[i]];
    }

    // new tangent
    for (int i=0; i<5; i++)
    {
        C[mapim[i]][iz      ] = 0.0;
	C[iz      ][mapim[i]] = 0.0;
    }
	C[iz][iz] = 0.0;

    for (int i=0; i<5; i++)
      for (int j=0; j<5; j++)
	C[mapim[i]][mapim[j]] = C[mapim[i]][mapim[j]] - Cmz[i]*Czm[j]/Czz;		
	
}




void reduced1zFMP :: secondPiolaKirchhoffStressVector(double S[6]) const
{
    theFiniteStrainMP -> secondPiolaKirchhoffStressVector(S);
}




void reduced1zFMP :: updateCurrentState(const double theTime, itensor& F)
{
    theFiniteStrainMP -> updateCurrentState(theTime, F);

    F = theFiniteStrainMP -> deformationGradient();
    const int iz = mapiz[0];
    // get R
    istensor FtF;
    FtF = istensor::tensorTransposedTimesTensor(F);
    istensor Uc;
    Uc = istensor::squareRoot(FtF);
    itensor R;
    R = F*Uc.inverse();
		
    double S[6];
    theFiniteStrainMP -> secondPiolaKirchhoffStressVector(S);
    double Sz;
    Sz = S[iz]; 
    double normSz;
    normSz = std::sqrt(Sz*Sz);
        
    // set constants for iteration loop
    double initnormS = 0.0;
    for (int i=0; i<6; i++)
        initnormS += S[i]*S[i];
    
    double minSz = tol*std::sqrt(initnormS);
    int iter = 0;
    double Ecz, Czz, Ezi;
    double Cc[6][6];
    istensor Ec;
    while ( (normSz>minSz) && (iter<maxiter) )
    {
        F = theFiniteStrainMP -> deformationGradient();
        Ec = 0.5 * ( istensor::tensorTransposedTimesTensor(F) - istensor::identity() );
	Ecz = coefE[iz] * Ec(voigt(0,iz),voigt(1,iz));
	theFiniteStrainMP -> convectedTangentMatrix(Cc);

	Czz = Cc[iz][iz];
	if (std::abs(Czz)>=tol)
	{
	    Ezi = Ecz - Sz/Czz;
	}
	else
	{
	    break;
	}
	// new F
	Ec(voigt(0,iz),voigt(1,iz)) = Ec(voigt(1,iz), voigt(0,iz)) = Ezi/coefE[iz];
	FtF = 2.0*Ec + istensor::identity();
	Uc = istensor::squareRoot(FtF);
	F = R*Uc;

        theFiniteStrainMP -> updateCurrentState(theTime, F);

	// Sz
	theFiniteStrainMP -> secondPiolaKirchhoffStressVector(S);
	Sz = S[iz];
	normSz = std::sqrt(Sz*Sz);
	iter++;
    }
     
    if (iter>=maxiter) std::cout << " Reduced model (1 zero model) has not converged, normSz = " << normSz << std::endl;	
}




/*********************************************************************************
  reduced3zFMP
**********************************************************************************/

reduced3zFMP :: reduced3zFMP(finiteStrainMP *mp, int inmapim[3], int inmapiz[3]) :
muesli::rFiniteStrainMP(mp)
{
    for (int i=0; i<3; i++)
        mapim[i] = inmapim[i];

    for (int i=0; i<3; i++)
        mapiz[i] = inmapiz[i];
} 




reduced3zFMP :: ~reduced3zFMP()
{

}




void reduced3zFMP :: convectedTangentMatrix(double C[6][6]) const
{
    itensor4 tg;
    theFiniteStrainMP -> convectedTangent(tg);
    muesli::tensorToMatrix(tg, C);

    itensor Czz;
    for (int i=0; i<3; i++)
      for (int j=0; j<3; j++)
        Czz(i,j) = C[mapiz[i]][mapiz[j]];
    itensor Cmz;
    for (int i=0; i<3; i++)
      for (int j=0; j<3; j++)
	Cmz(i,j) = C[mapim[i]][mapiz[j]];
    itensor Czm;
    for (int i=0; i<3; i++)
      for (int j=0; j<3; j++)
	Czm(i,j) = C[mapiz[i]][mapim[j]];
    itensor Cmm;
    for (int i=0; i<3; i++)
      for (int j=0; j<3; j++)
        Cmm(i,j) = C[mapim[i]][mapim[j]];

    itensor extraTg;
    extraTg = Cmz * Czz.inverse() * Czm;

    // new tangent
    for (int i=0; i<6; i++)
      for (int j=0; j<6; j++)
	C[i][j] = 0.0;
	
    for (int i=0; i<3; i++)
      for (int j=0; j<3; j++)
	C[mapim[i]][mapim[j]] = Cmm(i,j) - extraTg(i,j);

}




void reduced3zFMP :: secondPiolaKirchhoffStressVector(double S[6]) const
{
    theFiniteStrainMP -> secondPiolaKirchhoffStressVector(S);
}




void reduced3zFMP :: updateCurrentState(const double theTime, itensor& F)
{
    theFiniteStrainMP -> updateCurrentState(theTime, F);

    F = theFiniteStrainMP -> deformationGradient();
    // get R
    istensor FtF;
    FtF = istensor::tensorTransposedTimesTensor(F);
    istensor Uc;
    Uc = istensor::squareRoot(FtF);
    itensor R;
    R = F*Uc.inverse();
    
    double S[6];
    theFiniteStrainMP -> secondPiolaKirchhoffStressVector(S);
    ivector Sz(S[mapiz[0]], S[mapiz[1]], S[mapiz[2]]);
    double normSz;
    normSz = Sz.norm();
       
    // set constants for iteration loop
    double initnormS = 0.0;
    for (int i=0; i<6; i++)
        initnormS += S[i]*S[i];
   
    double minSz = tol * std::sqrt(initnormS);
    int iter = 0;
    ivector Ecz, Ezi;
    itensor Czz;
    Czz.setZero();
    double Cc[6][6];
    istensor Ec;
		
    while ( (normSz>minSz) && (iter<maxiter) )
    {
        F = theFiniteStrainMP -> deformationGradient();

        Ec = 0.5 * ( istensor::tensorTransposedTimesTensor(F) - istensor::identity() );
     
	for (int i=0; i<3; i++)
        	Ecz(i) = coefE[mapiz[i]] * Ec(voigt(0,mapiz[i]),voigt(1,mapiz[i]));

        theFiniteStrainMP -> convectedTangentMatrix(Cc);
     			
        for (int i=0; i<3; i++)
	  for (int j=0; j<3; j++)
	    Czz(i,j) = Cc[mapiz[i]][mapiz[j]];
			
        if (Czz.determinant()>=tol)
	{
	    Ezi = Ecz - Czz.inverse()*Sz;
	}
	else
	{
	    break;
	}
	// new F
	for (int i=0; i<3; i++)
	    Ec(voigt(0,mapiz[i]), voigt(1,mapiz[i])) = Ec(voigt(1,mapiz[i]),voigt(0,mapiz[i])) = Ezi(i)/coefE[mapiz[i]];
        FtF = 2.0*Ec + istensor::identity();
	Uc = istensor::squareRoot(FtF);
	F = R*Uc;
        theFiniteStrainMP -> updateCurrentState(theTime, F);
	
        // Sz
	theFiniteStrainMP -> secondPiolaKirchhoffStressVector(S);
        for (int i=0; i<3; i++)
	    Sz(i) = S[mapiz[i]];	
			
	normSz = Sz.norm();
	iter++;

    }
    
    if (iter>=maxiter) std::cout << " Reduced model (3 zero model) has not converged, normSz = " << normSz  << std::endl;	
}




/*********************************************************************************
  fbeam
**********************************************************************************/

int mapim_fbeam[3] = {0, 4, 5};
int mapiz_fbeam[3] = {1, 2, 3};

fbeamMP :: fbeamMP(finiteStrainMP *mp) :
muesli::reduced3zFMP(mp,mapim_fbeam,mapiz_fbeam)
{
} 




fbeamMP :: ~fbeamMP()
{

}




/*********************************************************************************
  fshell
**********************************************************************************/

int mapim_fshell[5] = {0, 1, 3, 4, 5};
int mapiz_fshell[1] = {2};

fshellMP :: fshellMP(finiteStrainMP *mp) :
muesli::reduced1zFMP(mp,mapim_fshell,mapiz_fshell)
{
} 




fshellMP :: ~fshellMP()
{

}
