/****************************************************************************
 *
 *                                 M U E S L I   v 1.4
 *
 *
 *     Copyright 2016 IMDEA Materials Institute, Getafe, Madrid, Spain
 *     Contact: muesli.materials@imdea.org
 *     Author: Ignacio Romero (ignacio.romero@imdea.org)
 *
 *     This file is part of MUESLI.
 *
 *     MUESLI is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     MUESLI is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with MUESLI.  If not, see <http://www.gnu.org/licenses/>.
 *
*****************************************************************************/


#include <stdio.h>
#include "thermofinitestrain.h"
#include "muesli/Finitestrain/finitestrainlib.h"

#define THETA0 100.0

using namespace muesli;

thermofiniteStrainMaterial :: thermofiniteStrainMaterial(const std::string& name,
                                                         const materialProperties& cl)
:
material(name, cl),
theFSMaterial(nullptr),
_rho(0.0),
_heatSupply(0.0),
_thermalExpansion(0.0),
_thermalCapacity(0.0),
_thermalConductivity(0.0),
_ref_temp(THETA0)
{
    //if      (cl.find("mechanical neohookean") != cl.end()) theFSMaterial = new neohookeanMaterial(name, cl);
    //else if (cl.find("mechanical svk") != cl.end())        theFSMaterial = new svkMaterial(name, cl);

    theFSMaterial = new neohookeanMaterial(name, cl);

    muesli::assignValue(cl, "density", _rho);
    muesli::assignValue(cl, "heat_supply", _heatSupply);
    muesli::assignValue(cl, "thermal_expansion", _thermalExpansion);
    muesli::assignValue(cl, "heat_capacity", _thermalCapacity);
    muesli::assignValue(cl, "conductivity", _thermalConductivity);
    muesli::assignValue(cl, "reference_temperature", _ref_temp);
}




thermofiniteStrainMP :: thermofiniteStrainMP(const thermofiniteStrainMaterial& m) :
thethermoFiniteStrainMaterial(m),
theFSMP(nullptr)
{
    theFSMP=m.theFSMaterial->createMaterialPoint();

    temp_c = m.referenceTemperature();
    temp_n = temp_c;
    gradT_n.setZero();
    gradT_c.setZero();
}




thermofiniteStrainMP* thermofiniteStrainMaterial :: createMaterialPoint() const
{
    muesli::thermofiniteStrainMP* mp = new thermofiniteStrainMP(*this);
    return mp;
}




bool thermofiniteStrainMaterial :: check() const
{
    bool ret = true;
    
    if (_thermalConductivity <= 0.0)
    {
        ret = false;
    }

    return ret;
}




double thermofiniteStrainMaterial :: density() const
{
    return theFSMaterial->density();
}




double thermofiniteStrainMaterial :: getProperty(const propertyName p) const
{
    double ret=0.0;
    
    // scan all the possible data
    switch (p)
    {
        case PR_CONDUCTIVITY: ret = _thermalConductivity; break;
        case PR_THERMAL_EXP:  ret = _thermalExpansion; break;
        case PR_THERMAL_CAP:  ret = _thermalCapacity; break;
        case PR_HEAT_SUPPLY:  ret = _heatSupply; break;

        default:
            ret = theFSMaterial->getProperty(p); //en el default
    }
    return ret;
}




void thermofiniteStrainMaterial :: print(std::ostream &of) const
{
    theFSMaterial->print(of);
    
    of  << "\n   Conductivity : " << _thermalConductivity;
    of  << "\n   Heat supply : " << _heatSupply;
    of  << "\n   Thermal expansion coef. : " << _thermalExpansion;
    of  << "\n   Thermal capacity : " << _thermalCapacity;
    of  << "\n";
}




const double& thermofiniteStrainMaterial :: referenceTemperature() const
{
    return _ref_temp;
}



void thermofiniteStrainMaterial :: setRandom()
{
    theFSMaterial->setRandom();
    
    _thermalConductivity = muesli::randomUniform(1.0, 10.0);
    _thermalExpansion    = muesli::randomUniform(1.0, 10.0);
    _thermalCapacity     = muesli::randomUniform(1.0, 10.0);
    _ref_temp            = muesli::randomUniform(10.0, 100.0);
}




bool thermofiniteStrainMaterial:: test(std::ostream &of)
{
    bool isok = true;
    setRandom();
    muesli::thermofiniteStrainMP* p = this->createMaterialPoint();
    
    isok = p->testImplementation(of);
    return isok;
}



void thermofiniteStrainMP:: CauchyStress(istensor& sigma) const
{
    theFSMP->CauchyStress(sigma);
    
    itensor& Fc = theFSMP->deformationGradient();
    itensor Finv =  theFSMP->deformationGradient().inverse();
    istensor Cinv = istensor :: tensorTimesTensorTransposed(Finv);
    double J = Fc.determinant();
    double Jinv = 1/J;
    
    const double deltaTemp = temp_c - thethermoFiniteStrainMaterial.referenceTemperature();
    const double bulk = thethermoFiniteStrainMaterial.theFSMaterial->getProperty(PR_BULK);

    istensor St = - 3.0 * thethermoFiniteStrainMaterial._thermalExpansion * bulk * deltaTemp * Cinv;
    sigma += Jinv * istensor::FSFt(Fc, St);
}




double thermofiniteStrainMaterial :: waveVelocity() const
{
    return theFSMaterial->waveVelocity();
}




void thermofiniteStrainMP :: CauchyStressVector(double S[6]) const
{
    istensor sigma;
    CauchyStress(sigma);
    muesli::tensorToVector(sigma, S);
}




void thermofiniteStrainMP :: commitCurrentState()
{
    theFSMP->commitCurrentState();
    
    gradT_n = gradT_c;
    temp_n = temp_c;
}




void thermofiniteStrainMP :: contractWithConvectedTangent(const ivector& v1, const ivector& v2, itensor& T) const
{
    theFSMP->contractWithConvectedTangent(v1, v2, T);
}




void thermofiniteStrainMP :: contractWithSpatialTangent(const ivector& v1, const ivector& v2, itensor& T) const
{
    theFSMP->contractWithSpatialTangent(v1, v2, T);
}




void thermofiniteStrainMP :: contractWithAllTangents(const ivector &v1,
                                                    const ivector& v2,
                                                    itensor&  Tdev,
                                                    istensor& Tmixed,
                                                    double&   Tvol) const
{
    theFSMP->contractWithAllTangents(v1, v2, Tdev, Tmixed, Tvol);
}




void thermofiniteStrainMP :: contractTangent(const ivector& na, const ivector& nb, double& tg) const
{
    tg = thethermoFiniteStrainMaterial._thermalConductivity * na.dot(nb);
}




void thermofiniteStrainMP :: convectedTangent(itensor4& ctg) const
{
    theFSMP->convectedTangent(ctg);

    itensor& Fc = theFSMP->deformationGradient();
    istensor C = istensor::tensorTransposedTimesTensor(Fc);
    istensor Cinv = C.inverse();

    const double alpha = thethermoFiniteStrainMaterial._thermalExpansion;
    const double bulk  = thethermoFiniteStrainMaterial.theFSMaterial->getProperty(PR_BULK);
    const double dtheta= temp_c - thethermoFiniteStrainMaterial.referenceTemperature();
    const double f     = 6.0 * alpha * bulk * dtheta;
    
    for (unsigned i=0; i<3; i++)
        for (unsigned j=0; j<3; j++)
            for (unsigned k=0; k<3; k++)
                for (unsigned l=0; l<3; l++)
                {
                    ctg(i,j,k,l) += f * 0.5 * ( Cinv(i,k)* Cinv(j,l) + Cinv(i,l)* Cinv(j,k) );
                }
}




void thermofiniteStrainMP :: convectedTangentMatrix(double C[6][6]) const
{
    itensor4 ct;
    convectedTangent(ct);
    muesli::tensorToMatrix(ct, C);
}




void thermofiniteStrainMP :: convectedTangentTimesSymmetricTensor(const istensor &M, istensor &CM) const
{
    theFSMP->convectedTangentTimesSymmetricTensor(M, CM);
}



double thermofiniteStrainMP :: dissipatedEnergy() const
{
    return theFSMP->dissipatedEnergy();
}




double thermofiniteStrainMP :: effectiveFreeEnergy() const
{
    itensor& Fc = theFSMP->deformationGradient();
    double J    = Fc.determinant();
    double logJ = log(J);

    const double alpha  = thethermoFiniteStrainMaterial._thermalExpansion;
    const double c0     = thethermoFiniteStrainMaterial._thermalCapacity;
    const double bulk   = thethermoFiniteStrainMaterial.theFSMaterial->getProperty(PR_BULK);
    const double theta0 = thethermoFiniteStrainMaterial.referenceTemperature();
    const double deltaTemp = temp_c - theta0;

    double Psi_coup = - 3.0 * alpha * bulk * deltaTemp * logJ;
    double Psi_ther = c0 * (deltaTemp - temp_c * log(temp_c/theta0));

    return theFSMP->effectiveStoredEnergy() + Psi_coup + Psi_ther;
}




void thermofiniteStrainMP :: energyMomentumTensor(const itensor &F, itensor &T) const
{
    theFSMP->energyMomentumTensor(F,T);
}




double thermofiniteStrainMP :: entropy() const
{
    itensor& Fc = theFSMP->deformationGradient();
    double J = Fc.determinant();
    double logJ = log(J);
    
    const double alpha = thethermoFiniteStrainMaterial._thermalExpansion;
    const double bulk  = thethermoFiniteStrainMaterial.theFSMaterial->getProperty(PR_BULK);
    const double c0    = thethermoFiniteStrainMaterial._thermalCapacity;
    const double theta0= thethermoFiniteStrainMaterial.referenceTemperature();
    return 3.0 * alpha * bulk * logJ + c0 * log(temp_c/theta0);
}




void thermofiniteStrainMP :: firstPiolaKirchhoffStress(itensor &P) const
{
    itensor& Fc = theFSMP->deformationGradient();
    istensor S;
    secondPiolaKirchhoffStress(S);
    P = Fc*S;
}




double thermofiniteStrainMP :: freeEnergy() const
{
    itensor& Fc = theFSMP->deformationGradient();
    double J    = Fc.determinant();
    double logJ = log(J);
    
    const double alpha = thethermoFiniteStrainMaterial._thermalExpansion;
    const double c0    = thethermoFiniteStrainMaterial._thermalCapacity;
    const double bulk  = thethermoFiniteStrainMaterial.theFSMaterial->getProperty(PR_BULK);
    const double theta0= thethermoFiniteStrainMaterial.referenceTemperature();
    const double deltaTemp   = temp_c - theta0;
    
    double Psi_coup = - 3.0 * alpha * bulk * deltaTemp * logJ;
    double Psi_ther = c0 * (deltaTemp - temp_c * log(temp_c/theta0));
    
    return theFSMP->storedEnergy() + Psi_coup + Psi_ther;
}




double thermofiniteStrainMP :: freeEntropy() const
{
    return freeEnergy()/temp_c;
}





materialState thermofiniteStrainMP :: getConvergedState() const
{
    materialState state;
    
    state = theFSMP->getConvergedState();
    state.theTime = time_n;
    state.theDouble.push_back(temp_n);
    state.theVector.push_back(gradT_n);
    
    return state;
}




materialState thermofiniteStrainMP :: getCurrentState() const
{
    materialState state;
    
    state = theFSMP->getCurrentState();
    state.theTime = time_c;
    state.theDouble.push_back(temp_c);
    state.theVector.push_back(gradT_c);
    
    return state;
}



// c = - theta d^2[Psi]/d[theta]^2
double thermofiniteStrainMP :: heatCapacity() const
{
    return thethermoFiniteStrainMaterial._thermalCapacity;
}




double thermofiniteStrainMP :: internalEnergy() const
{
    return freeEnergy() + temp_c * entropy();
}




void thermofiniteStrainMP :: KirchhoffStress(istensor &tau) const
{
    theFSMP->KirchhoffStress(tau);
}




void thermofiniteStrainMP :: KirchhoffStressVector(double tauv[6]) const
{
    theFSMP->KirchhoffStressVector(tauv);
}




istensor thermofiniteStrainMP :: materialConductivity() const
{
    return thethermoFiniteStrainMaterial._thermalConductivity * istensor::identity();
}




// coupling tensor M = theta * d^2(psi)/( dF dTheta )
itensor thermofiniteStrainMP :: materialCouplingTensor() const
{
    itensor& Fc = theFSMP->deformationGradient();
    return Fc * symmetricCouplingTensor();
}




itensor thermofiniteStrainMP :: materialCouplingTensorDTheta() const
{
    itensor H = materialCouplingTensor();
    return H * (1.0/temp_c);
}




ivector thermofiniteStrainMP :: materialHeatflux() const
{
    return -thethermoFiniteStrainMaterial._thermalConductivity * gradT_c;
}




void thermofiniteStrainMP :: materialTangent(itensor4& cm) const
{
    const itensor& Fc = theFSMP->deformationGradient();
    istensor S;
    secondPiolaKirchhoffStress(S);

    itensor4 cc;
    convectedTangent(cc);

    cm.setZero();
    for (unsigned a=0; a<3; a++)
        for (unsigned b=0; b<3; b++)
            for (unsigned A=0; A<3; A++)
                for (unsigned B=0; B<3; B++)
                {
                    if (a == b) cm(a,A,b,B) += S(A,B);

                    for (unsigned C=0; C<3; C++)
                        for (unsigned D=0; D<3; D++)
                            cm(a,A,b,B) += Fc(a,C) * Fc(b,D) * cc(C,A,D,B);
                }
}




double thermofiniteStrainMP :: plasticSlip() const
{
    return theFSMP->plasticSlip();
}




void thermofiniteStrainMP :: resetCurrentState()
{
    theFSMP->resetCurrentState();
}




void thermofiniteStrainMP :: secondPiolaKirchhoffStress(istensor& S) const
{
    theFSMP->secondPiolaKirchhoffStress(S);

    itensor  Finv = theFSMP->deformationGradient().inverse();
    istensor Cinv = istensor::tensorTimesTensorTransposed(Finv);

    const double alpha = thethermoFiniteStrainMaterial._thermalExpansion;
    const double bulk  = thethermoFiniteStrainMaterial.theFSMaterial->getProperty(PR_BULK);
    const double theta0= thethermoFiniteStrainMaterial.referenceTemperature();
    const double dTemp = temp_c - theta0;

    S -= 3.0 * alpha * bulk * dTemp * Cinv;
}




void  thermofiniteStrainMP :: secondPiolaKirchhoffStressVector(double Sv[6]) const
{
    istensor S;
    secondPiolaKirchhoffStress(S);
    muesli::tensorToVector(S, Sv);
}




istensor thermofiniteStrainMP :: spatialConductivity() const
{
    itensor& Fc = theFSMP->deformationGradient();
    double J = Fc.determinant();
    double Jinv = 1/J;
    istensor k = thethermoFiniteStrainMaterial._thermalConductivity*istensor::identity();
    return Jinv * istensor :: FSFt(Fc, k);
}




ivector thermofiniteStrainMP :: spatialHeatflux() const
{
    itensor& Fc = theFSMP->deformationGradient();
    double J = Fc.determinant();
    return - (1.0/J) * Fc * 0.5 * thethermoFiniteStrainMaterial._thermalConductivity * gradT_c/temp_c;
}

 
 

void thermofiniteStrainMP :: spatialTangentMatrix(double c[6][6]) const
{
    theFSMP->spatialTangentMatrix(c);
}




// coupling tensor M = 2.0 * theta * d^2(psi)/( dC dTheta ) = theta * d[S]/d[Theta]
istensor thermofiniteStrainMP :: symmetricCouplingTensor() const
{
    itensor& Fc = theFSMP->deformationGradient();
    istensor C  = istensor::tensorTransposedTimesTensor(Fc);
    istensor Cinv = C.inverse();

    const double alpha = thethermoFiniteStrainMaterial._thermalExpansion;
    const double bulk  = thethermoFiniteStrainMaterial.theFSMaterial->getProperty(PR_BULK);

    return -3.0 * alpha * bulk * temp_c * Cinv;
}



// D = 4 d^2[psi]/d C^2
itensor4 thermofiniteStrainMP :: symmetricCouplingTensorDC() const
{
    itensor& Fc = theFSMP->deformationGradient();
    istensor C = istensor::tensorTransposedTimesTensor(Fc);
    istensor Cinv = C.inverse();
    itensor4 D;

    const double alpha = thethermoFiniteStrainMaterial._thermalExpansion;
    const double bulk  = thethermoFiniteStrainMaterial.theFSMaterial->getProperty(PR_BULK);
    const double f     = 6.0 * alpha * bulk * temp_c;

    D.setZero();
    for (unsigned i=0; i<3; i++)
        for (unsigned j=0; j<3; j++)
            for (unsigned k=0; k<3; k++)
                for (unsigned l=0; l<3; l++)
                {
                    D(i,j,k,l) += f * 0.5 * ( Cinv(i,k)* Cinv(j,l) + Cinv(i,l)* Cinv(j,k) );
                }
    return D;
}




istensor thermofiniteStrainMP :: symmetricCouplingTensorDTheta() const
{
    istensor Hs = symmetricCouplingTensor();
    return Hs * (1.0/temp_c);
}




double& thermofiniteStrainMP :: temperature()
{
    return temp_c;
}




const double& thermofiniteStrainMP :: temperature() const
{
    return temp_c;
}




bool thermofiniteStrainMP :: testImplementation(std::ostream& of, const bool testDE, const bool testDDE) const
{
    bool isok = true;
    const double inc = 1.0e-4;
    thermofiniteStrainMP& theMP = const_cast<thermofiniteStrainMP&>(*this);

    // set a random update in the material
    itensor F;  F.setRandom();
    if (F.determinant() < 0.0) F *= -1.0;
    ivector gradT; gradT.setRandom();
    double  temp; temp = randomUniform(0.8, 1.2) * thethermoFiniteStrainMaterial.referenceTemperature();

    theMP.updateCurrentState(0.0, F, gradT, temp);
    theMP.commitCurrentState();

    double tn1 = muesli::randomUniform(0.1,1.0);
    F.setRandom();
    if (F.determinant() < 0.0) F *= -1.0;
    gradT.setRandom();
    temp = randomUniform(0.8, 1.2) * thethermoFiniteStrainMaterial.referenceTemperature();
    theMP.updateCurrentState(tn1, F, gradT, temp);

    // check programmed quantities by numerically differentiating the free energy
    theMP.resetCurrentState();


    // Derivatives with respect to F
    // Compute numerical value of P, 1PK stress tensor, = d Psi / d F
    // Compute numerical value of A, material tangent A_{iAjB} = d (P_iA) / d F_jB
    itensor num_P;
    itensor4 num_A;
    {
        itensor dP, Pp1, Pp2, Pm1, Pm2;

        for (size_t i=0; i<3; i++)
        {
            for (size_t j=0; j<3; j++)
            {
                const double original = F(i,j);

                F(i,j) = original + inc;
                theMP.updateCurrentState(tn1, F, gradT, temp);
                double Wp1 = effectiveFreeEnergy();
                firstPiolaKirchhoffStress(Pp1);

                F(i,j) = original + 2.0*inc;
                theMP.updateCurrentState(tn1, F, gradT, temp);
                double Wp2 = effectiveFreeEnergy();
                firstPiolaKirchhoffStress(Pp2);

                F(i,j) = original - inc;
                theMP.updateCurrentState(tn1, F, gradT, temp);
                double Wm1 = effectiveFreeEnergy();
                firstPiolaKirchhoffStress(Pm1);

                F(i,j) = original - 2.0*inc;
                theMP.updateCurrentState(tn1, F, gradT, temp);
                double Wm2 = effectiveFreeEnergy();
                firstPiolaKirchhoffStress(Pm2);

                // fourth order approximation of the derivative
                num_P(i,j) = (-Wp2 + 8.0*Wp1 - 8.0*Wm1 + Wm2)/(12.0*inc);

                // derivative of PK stress
                dP = (-Pp2 + 8.0*Pp1 - 8.0*Pm1 + Pm2)/(12.0*inc);
                for (unsigned k=0; k<3; k++)
                    for (unsigned l=0; l<3; l++)
                        num_A(k,l,i,j) = dP(k,l);

                F(i,j) = original;
                theMP.updateCurrentState(tn1, F, gradT, temp);
            }
        }
    }


    // Derivatives with respect to theta
    // Compute numerical value of entropy = - d Psi / d theta
    // Compute numerical value of coupling tensor M = theta * d P / d theta
    // Compute numerical value of heat capacity =  theta * (d s / d theta)
    // Compute numerical value of d M / d theta
    double num_entropy;
    itensor num_coupling;
    itensor num_dMdt;
    double num_c;
    {
        const double original = temp;

        temp = original + inc;
        theMP.updateCurrentState(tn1, F, gradT, temp);
        double Psip1 = effectiveFreeEnergy();
        itensor PK_p1; firstPiolaKirchhoffStress(PK_p1);
        double s_p1 = entropy();
        itensor M_p1 = materialCouplingTensor();

        temp = original + 2.0*inc;
        theMP.updateCurrentState(tn1, F, gradT, temp);
        double Psip2 = effectiveFreeEnergy();
        itensor PK_p2; firstPiolaKirchhoffStress(PK_p2);
        double s_p2 = entropy();
        itensor M_p2 = materialCouplingTensor();

        temp = original - inc;
        theMP.updateCurrentState(tn1, F, gradT, temp);
        double Psim1 = effectiveFreeEnergy();
        itensor PK_m1; firstPiolaKirchhoffStress(PK_m1);
        double s_m1 = entropy();
        itensor M_m1 = materialCouplingTensor();

        temp = original - 2.0*inc;
        theMP.updateCurrentState(tn1, F, gradT, temp);
        double Psim2 = effectiveFreeEnergy();
        itensor PK_m2; firstPiolaKirchhoffStress(PK_m2);
        double s_m2 = entropy();
        itensor M_m2 = materialCouplingTensor();

        temp = original;
        theMP.updateCurrentState(tn1, F, gradT, temp);

        double theta = temp_c;

        // fourth order approximation of the derivative
        num_entropy  = -(-Psip2 + 8.0*Psip1 - 8.0*Psim1 + Psim2)/(12.0*inc);
        num_coupling = theta * (-PK_p2 + 8.0*PK_p1 - 8.0*PK_m1 + PK_m2)/(12.0*inc);
        num_c        = theta * (-s_p2 + 8.0*s_p1 - 8.0*s_m1 + s_m2)/(12.0*inc);
        num_dMdt     = (-M_p2 + + 8.0*M_p1 - 8.0*M_m1 + M_m2)/(12.0*inc);
    }


    // compare 1st PK stress with derivative of free energy wrt F
    if (testDE)
    {
        itensor pr_P;
        firstPiolaKirchhoffStress(pr_P);

        itensor errorP = num_P - pr_P;
        isok = (errorP.norm()/pr_P.norm() < 1e-4);
        of << "\n   1. Comparing P with derivative [d Psi / d F].";
        if (isok)
        {
            of << " Test passed.";
        }
        else
        {
            of << "\n Relative error in 1st PK computation %e. Test failed." << errorP.norm()/pr_P.norm();
            of << "\n " << pr_P;
            of << "\n " << num_P;
        }
    }
    else
    {
        of << "\n   1. Comparing P with derivative of Psi ::: not run for this material";
    }


    // compare entropy with derivative of Psi wrt theta (-)
    {
        double pr_entropy = entropy();

        double error = num_entropy - pr_entropy;
        isok = (fabs(error)/fabs(pr_entropy) < 1e-4);
        of << "\n   2. Comparing entropy with  [- d Psi / d theta ].";
        if (isok)
        {
            of << " Test passed.";
        }
        else
        {
            of << "\n Relative error entropy computation %e. Test failed." << fabs(error)/fabs(pr_entropy);
            of << "\n " << pr_entropy;
            of << "\n " << num_entropy;
        }
    }


    // compare material tangent with derivative of the 1st PK w.r.t. F
    if (testDDE)
    {
        // programmed material tangent
        itensor4 pr_A;
        materialTangent(pr_A);

        // relative error
        itensor4 errorA = num_A - pr_A;
        double error = errorA.norm();
        double norm = pr_A.norm();
        isok  = (error/norm < 1e-4);

        of << "\n   3. Comparing material tangent with [d 1PK / d F ].";
        if (isok)
        {
            of << " Test passed.";
        }
        else
        {
            of << "\n      Test failed.";
            of << "\n      Relative error in DStress computation: " <<  error/norm;
            of << errorA;
        }
    }


    // test coupling tensor with theta * d P / d theta
    {
        itensor pr_coupling = materialCouplingTensor();

        itensor error = num_coupling - pr_coupling;
        isok = (error.norm()/pr_coupling.norm() < 1e-4);
        of << "\n   4. Comparing M with [theta d P / d theta].";
        if (isok)
        {
            of << " Test passed.";
        }
        else
        {
            of << "\n Relative error in coupling tensor computation. Test failed." << error.norm()/pr_coupling.norm();
            of << "\n " << pr_coupling;
            of << "\n " << num_coupling;
        }
    }


    // compare heat capacity with temperature * derivative of entropy wrt theta
    {
        double pr_c = heatCapacity();

        double error = num_c - pr_c;
        isok = (fabs(error)/fabs(pr_c) < 1e-4);
        of << "\n   5. Comparing heat capacity with  [theta * d s / d theta].";
        if (isok)
        {
            of << " Test passed.";
        }
        else
        {
            of << "\n Relative error c computation %e. Test failed." << fabs(error)/fabs(pr_c);
            of << "\n " << pr_c;
            of << "\n " << num_c;
        }
    }


    // compare derivative of coupling tensor
    {
        itensor dMdt = materialCouplingTensorDTheta();
        itensor error = num_dMdt - dMdt;
        isok = (error.norm()/dMdt.norm() < 1e-4);

        of << "\n   6. Comparing DM/dtheta with derivative of M.";
        if (isok)
        {
            of << " Test passed.";
        }
        else
        {
            of << "\n Relative error in DM/Dtheta. Test failed." << error.norm()/dMdt.norm();
            of << "\n " << dMdt;
            of << "\n " << num_dMdt;
        }
    }

    return isok;
}




double thermofiniteStrainMP :: thermalPotential() const
{
    return 0.5* thethermoFiniteStrainMaterial._thermalConductivity * (gradT_c/temp_c).squaredNorm();
}




double thermofiniteStrainMP :: volumetricStiffness() const
{
    return theFSMP->volumetricStiffness();
}




void thermofiniteStrainMP :: updateCurrentState(const double theTime, const itensor& F, const ivector& gradT, const double& temp)
{
    theFSMP->updateCurrentState(theTime, F);
    
    gradT_c = gradT;
    temp_c  = temp;
}



