/****************************************************************************
 *
 *                                 M U E S L I   v 1.4
 *
 *
 *     Copyright 2016 IMDEA Materials Institute, Getafe, Madrid, Spain
 *     Contact: muesli.materials@imdea.org
 *     Author: Ignacio Romero (ignacio.romero@imdea.org)
 *
 *     This file is part of MUESLI.
 *
 *     MUESLI is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     MUESLI is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with MUESLI.  If not, see <http://www.gnu.org/licenses/>.
 *
*****************************************************************************/






#include "reducedsmallstrain.h"

#include <iostream>
#include <cmath>

/*
 *
 * reducedsmallstrain.cpp
 * reduced models with zero stresses, such as shell and beams
 * D. Portillo, may 2017
 *
 * see "Using finite strain 3D-material models in beam and 
 *      shell elements", S. Klinkel & S. Govindjee
 *
 */


const int maxiter = 50;
const double tol = 1.0e-10;
const double coefE[6] ={1.0, 1.0, 1.0, 2.0, 2.0, 2.0};

using namespace muesli;


rSmallStrainMP :: rSmallStrainMP(smallStrainMP *mp)
{
    _theSmallStrainMP = const_cast<smallStrainMP*>(mp);
}




void rSmallStrainMP :: commitCurrentState()
{
    _theSmallStrainMP -> commitCurrentState();
}




double rSmallStrainMP :: pressure() const
{
	return _theSmallStrainMP -> pressure();
}




void rSmallStrainMP :: deviatoricStress(istensor& s) const
{
    istensor sigma;
    this->stress(sigma);
    s = istensor::deviatoricPart(sigma);
}




double rSmallStrainMP :: plasticSlip() const
{
	return _theSmallStrainMP -> plasticSlip();
}




double rSmallStrainMP :: deviatoricEnergy() const
{
    return _theSmallStrainMP -> deviatoricEnergy();
}




double rSmallStrainMP :: dissipatedEnergy() const
{
    return _theSmallStrainMP -> dissipatedEnergy();
}




double rSmallStrainMP :: effectiveStoredEnergy() const
{
    return _theSmallStrainMP -> effectiveStoredEnergy();
}




double rSmallStrainMP :: storedEnergy() const
{
    return _theSmallStrainMP -> storedEnergy();
}




smallStrainMP& rSmallStrainMP :: theSmallStrainMP()
{
    return *_theSmallStrainMP;
}




double rSmallStrainMP :: volumetricEnergy() const
{
    return _theSmallStrainMP -> volumetricEnergy();
}




materialState rSmallStrainMP :: getConvergedState() const
{
    materialState state_n;

	state_n = _theSmallStrainMP -> getConvergedState();

    return state_n;
}




materialState rSmallStrainMP :: getCurrentState() const
{
    materialState state_c;

	state_c = _theSmallStrainMP -> getCurrentState();

    return state_c;
}




istensor& rSmallStrainMP :: getCurrentStrain()
{
    return _theSmallStrainMP -> getCurrentStrain();
}




const istensor& rSmallStrainMP :: getCurrentStrain() const
{
    return _theSmallStrainMP -> getCurrentStrain();
}




void rSmallStrainMP :: resetCurrentState()
{
	_theSmallStrainMP -> resetCurrentState();
}




void rSmallStrainMP :: setRandom()
{
	_theSmallStrainMP -> setRandom();
}




void rSmallStrainMP :: stress(istensor& s) const
{
    _theSmallStrainMP -> stress(s);
}




bool rSmallStrainMP :: testImplementation(std::ostream& os) const
{
    bool isok = true;

    // set a random update in the material
    istensor eps;
    eps.setZero();

    rSmallStrainMP* ssmp = const_cast<rSmallStrainMP*>(this);
    ssmp->updateCurrentState(0.0, eps);
    ssmp->commitCurrentState();

    istensor epsaux;
    epsaux.setRandom();
    eps += 1.0e-2 * epsaux;
    double tn1 = muesli::randomUniform(0.1,1.0);
    //eps.setRandom();
    rSmallStrainMP& theMP = const_cast<rSmallStrainMP&>(*this);;
	
    theMP.updateCurrentState(tn1, eps);
  

    // (1)  compare DEnergy with the derivative of Energy
    // programmed stress
    istensor sigma;
    _theSmallStrainMP -> stress(sigma);
    // numerical differentiation stress
    istensor numSigma;
    numSigma.setZero();
    const double   inc = 1.0e-3*eps.norm();

    // numeric C
    itensor4 nC;
    nC.setZero();

    // numerical differentiation sigma
   istensor dsigma, sigmap1, sigmap2, sigmam1, sigmam2;

   for (size_t i=0; i<3; i++)
     {
     for (size_t j=i; j<3; j++)
       {
         double original = eps(i,j);

         eps(i,j) = eps(j,i) = original + inc;
         theMP.updateCurrentState(tn1, eps);
         stress(sigmap1);
         double Wp1 = effectiveStoredEnergy();

         eps(i,j) = eps(j,i) = original + 2.0*inc;
         theMP.updateCurrentState(tn1, eps);
         stress(sigmap2);
         double Wp2 = effectiveStoredEnergy();

         eps(i,j) = eps(j,i) = original - inc;
         theMP.updateCurrentState(tn1, eps);
         stress(sigmam1);
         double Wm1 = effectiveStoredEnergy();

         eps(i,j) = eps(j,i) = original - 2.0*inc;
         theMP.updateCurrentState(tn1, eps);
         stress(sigmam2);
         double Wm2 = effectiveStoredEnergy();

         // fourth order approximation of the derivative
         double der = (-Wp2 + 8.0*Wp1 - 8.0*Wm1 + Wm2)/(12.0*inc);
         numSigma(i,j) = der;
         if (i != j) numSigma(i,j) *= 0.5;

         numSigma(j,i) = numSigma(i,j);

         // fourth order approximation of the derivative
         dsigma = (-sigmap2 + 8.0*sigmap1 - 8.0*sigmam1 + sigmam2)/(12.0*inc);

         if (i != j) dsigma *= 0.5;


         for (unsigned k=0; k<3; k++)
           for (unsigned l=0; l<3; l++)
             nC(k,l,i,j) = nC(k,l,j,i) = dsigma(k,l);

         eps(i,j) = eps(j,i) = original;
         theMP.updateCurrentState(tn1, eps);
       }
     }

     // DE
     if (true)
     {
       // relative error less than 0.01%
       istensor error = numSigma - sigma;
       isok = (error.norm()/sigma.norm() < 1e-4);

       os << "\n   1. Comparing stress with DWeff (reduced model).";
       if (isok)
       {
         os << " Test passed.";
       }
       else
       {
         os << "\n      Test failed.";
         os << "\n      Relative error in DWeff computation: " <<  error.norm()/sigma.norm();
         os << "\n      Stress:\n" << sigma;
         os << "\n      Numeric stress:\n" << numSigma;
       }
     }

     // DDE
     if (true)
     {
       // get the voigt notation
       double nCv[6][6];
       muesli::tensorToMatrix(nC,nCv);
		
       // get the tangent in voigt notation
       double tgv[6][6];
       tangentMatrix(tgv);
		
       // relative error less than 0.01%
       double errordde = 0.0;
       double normdde = 0.0;
       for (unsigned i=0; i<6; i++)
         for (unsigned j=0; j<6; j++)
         {
           errordde += pow(nCv[i][j]-tgv[i][j],2);
           normdde  += pow(tgv[i][j],2);
         }
       errordde = sqrt(errordde);
       normdde = sqrt(normdde);
       isok = (errordde/normdde < 1e-4);

       os << "\n   2. Comparing tensor C with DStress (reduced model).";
       if (isok)
       {
         os << " Test passed.";
       }
       else
       {
         os << " nCv = " << std::endl;
         for (int i=0;i <6 ; i++)
	 {
           os << " [ " << nCv[i][0];
           for (int j=1; j<6; j++)
             os << " , " << nCv[i][j];
           os << " ] " << std::endl;
	 }	
         os << " tgv = " << std::endl;
         for (int i=0;i <6 ; i++)
	 {
	   os << " [ " << tgv[i][0];
	   for (int j=1; j<6; j++)
	     os << " , " << tgv[i][j];

	   os << " ] " << std::endl;
	 }	
         os << "\n      Test failed.";
         os << "\n      Relative error in DWeff computation: " <<  errordde/normdde;
       }
     }
	

     return isok;
}




/*********************************************************************************
  reduced1zSMP
**********************************************************************************/

reduced1zSMP :: reduced1zSMP(smallStrainMP *mp, int inmapim[5], int inmapiz[1]) :
muesli::rSmallStrainMP(mp)
{
    for (int i=0; i<5; i++)
        mapim[i] = inmapim[i];

    mapiz[0] = inmapiz[0];
} 




reduced1zSMP :: ~reduced1zSMP()
{

}




void reduced1zSMP :: tangentMatrix(double C[6][6]) const
{
    itensor4 tg;
    _theSmallStrainMP -> tangentTensor(tg);
    muesli::tensorToMatrix(tg, C);

    int iz = mapiz[0]; // Cc's index such that Cc[iz][iz] = Czz
    double Czz = C[iz][iz];
    double Cmz[5] = {0.0};
    double Czm[5] = {0.0};

    for (int i=0; i<5; i++)
    {
       Cmz[i] = C[mapim[i]][iz      ];
       Czm[i] = C[iz      ][mapim[i]];
    }

    // new tangent
    for (int i=0; i<5; i++)
    {
        C[mapim[i]][iz      ] = 0.0;
	C[iz      ][mapim[i]] = 0.0;
    }
	C[iz][iz] = 0.0;

    for (int i=0; i<5; i++)
      for (int j=0; j<5; j++)
	C[mapim[i]][mapim[j]] = C[mapim[i]][mapim[j]] - Cmz[i]*Czm[j]/Czz;		
	
}




void reduced1zSMP :: stressVector(double S[6]) const
{
    _theSmallStrainMP -> stressVector(S);
}




void reduced1zSMP :: updateCurrentState(const double theTime, istensor& strain)
{
    _theSmallStrainMP -> updateCurrentState(theTime, strain);

    strain = _theSmallStrainMP -> getCurrentStrain();

    const int iz = mapiz[0];

    double S[6];
    _theSmallStrainMP -> stressVector(S);
    double Sz; // shells
    Sz = S[iz]; // Sz = S33
    double normSz;
    normSz = std::sqrt(Sz*Sz);
    // set constants for iteration loop
    double initnormS = 0.0;
    for (int i=0; i<6; i++)
        initnormS += S[i]*S[i];
    double minSz = tol*std::sqrt(initnormS);
    int iter = 0;
    double Ecz, Czz, Ezi;
    double Cc[6][6];
    istensor Ec;

    while ( (normSz>minSz) && (iter<maxiter) )
    {
    	strain = _theSmallStrainMP -> getCurrentStrain();

        Ecz = coefE[iz] * strain(voigt(0,iz),voigt(1,iz));
	_theSmallStrainMP -> tangentMatrix(Cc);
	Czz = Cc[iz][iz];
	if (std::abs(Czz)>=tol)
	{
	    Ezi = Ecz - Sz/Czz;
	}
	else
	{
	    break;
	}
	// new Fc
	strain(voigt(0,iz),voigt(1,iz)) = strain(voigt(1,iz),voigt(0,iz)) = Ezi/coefE[iz];
	_theSmallStrainMP -> updateCurrentState(theTime, strain);
	// Sz
	_theSmallStrainMP -> stressVector(S);
	Sz = S[iz];
	normSz = std::sqrt(Sz*Sz);
	iter++;

     }

     if (iter>=maxiter) std::cout << " Reduced model (1 zero model) has not converged, normSz " << normSz << std::endl;	
}




/*********************************************************************************
  reduced3zSMP
**********************************************************************************/

reduced3zSMP :: reduced3zSMP(smallStrainMP *mp, int inmapim[3], int inmapiz[3]) :
muesli::rSmallStrainMP(mp)
{
    for (int i=0; i<3; i++)
        mapim[i] = inmapim[i];

    for (int i=0; i<3; i++)
        mapiz[i] = inmapiz[i];
} 




reduced3zSMP :: ~reduced3zSMP()
{

}




void reduced3zSMP :: tangentMatrix(double C[6][6]) const
{
    itensor4 tg;
    _theSmallStrainMP -> tangentTensor(tg);
    muesli::tensorToMatrix(tg, C);

    itensor Czz;
    for (int i=0; i<3; i++)
      for (int j=0; j<3; j++)
        Czz(i,j) = C[mapiz[i]][mapiz[j]];
    itensor Cmz;
    for (int i=0; i<3; i++)
      for (int j=0; j<3; j++)
	Cmz(i,j) = C[mapim[i]][mapiz[j]];
    itensor Czm;
    for (int i=0; i<3; i++)
      for (int j=0; j<3; j++)
	Czm(i,j) = C[mapiz[i]][mapim[j]];
    itensor Cmm;
    for (int i=0; i<3; i++)
      for (int j=0; j<3; j++)
        Cmm(i,j) = C[mapim[i]][mapim[j]];

    itensor extraTg;
    extraTg = Cmz * Czz.inverse() * Czm;

    // new tangent
    for (int i=0; i<6; i++)
      for (int j=0; j<6; j++)
	C[i][j] = 0.0;
	
    for (int i=0; i<3; i++)
      for (int j=0; j<3; j++)
	C[mapim[i]][mapim[j]] = Cmm(i,j) - extraTg(i,j);

}




void reduced3zSMP :: stressVector(double S[6]) const
{
    _theSmallStrainMP -> stressVector(S);
}




void reduced3zSMP :: updateCurrentState(const double theTime, istensor& strain)
{
    _theSmallStrainMP -> updateCurrentState(theTime, strain);
   
    strain = _theSmallStrainMP -> getCurrentStrain();

    double S[6];
    _theSmallStrainMP -> stressVector(S);
    ivector Sz(S[mapiz[0]], S[mapiz[1]], S[mapiz[2]]); // beams
    double normSz;
    normSz = Sz.norm();
    // set constants for iteration loop
    double initnormS = 0.0;
    for (int i=0; i<6; i++)
      initnormS += S[i]*S[i];
    double minSz = tol*std::sqrt(initnormS);
    int iter = 0;
    ivector Ecz, Ezi;
    itensor Czz;
    Czz.setZero();
    double Cc[6][6];
    istensor Ec;
    ivector Saux;

    while ( (normSz>minSz) && (iter<maxiter) )
    {
    	strain = _theSmallStrainMP -> getCurrentStrain();

	for (int i=0; i<3; i++)
        	Ecz(i) = coefE[mapiz[i]] * strain(voigt(0,mapiz[i]), voigt(1,mapiz[i]) );

        _theSmallStrainMP -> tangentMatrix(Cc);

	for (int i=0; i<3; i++)
	  for (int j=0; j<3; j++)
	    Czz(i,j) = Cc[mapiz[i]][mapiz[j]];
	

	if (Czz.determinant()>=tol)
	{
	    Ezi = Ecz - Czz.inverse()*Sz;
	}
	else
	{
	    break;
	}
        // new eps_c
	//strain = istensor(strain(0,0),Ezi(0),Ezi(1),0.5*Ezi(2),strain(0,2),strain(0,1));
	for (int i=0; i<3; i++)
	    strain(voigt(0,mapiz[i]), voigt(1,mapiz[i])) = strain(voigt(1,mapiz[i]),voigt(0,mapiz[i])) = Ezi(i)/coefE[mapiz[i]];
	_theSmallStrainMP -> updateCurrentState(theTime, strain);

	// Sz
	_theSmallStrainMP -> stressVector(S);
        for (int i=0; i<3; i++)
	    Sz(i) = S[mapiz[i]];	

	normSz = Sz.norm();
	iter++;

    }

    if (iter>=maxiter) std::cout << " Reduced model (3 zero model) has not converged, normSz " << normSz << std::endl;	

}




/*********************************************************************************
  sbeam
**********************************************************************************/

int mapim_beam[3] = {0, 4, 5};
int mapiz_beam[3] = {1, 2, 3};

sbeamMP :: sbeamMP(smallStrainMP *mp) :
muesli::reduced3zSMP(mp,mapim_beam,mapiz_beam)
{
} 





/*********************************************************************************
  sshell
**********************************************************************************/

int mapim_shell[5] = {0, 1, 3, 4, 5};
int mapiz_shell[1] = {2};

sshellMP :: sshellMP(smallStrainMP *mp) :
muesli::reduced1zSMP(mp,mapim_shell,mapiz_shell)
{
} 

