/****************************************************************************
 *
 *                                 M U E S L I   v 1.4
 *
 *
 *     Copyright 2016 IMDEA Materials Institute, Getafe, Madrid, Spain
 *     Contact: muesli.materials@imdea.org
 *     Author: Ignacio Romero (ignacio.romero@imdea.org)
 *
 *     This file is part of MUESLI.
 *
 *     MUESLI is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     MUESLI is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with MUESLI.  If not, see <http://www.gnu.org/licenses/>.
 *
*****************************************************************************/







#include "conductor.h"


using namespace std;
using namespace muesli;




conductorMaterial :: conductorMaterial(const std::string& name,
                                       const materialProperties& cl)
:
  material(name, cl),
  _conductivity(0.0),
  _density(0.0),
  _capacity(0.0)
{
    muesli::assignValue(cl, "conductivity", _conductivity);
    muesli::assignValue(cl, "density",      _density);
    muesli::assignValue(cl, "capacity",     _capacity);
}




conductorMaterial :: conductorMaterial(const std::string& name, const double k, const double rho)
:
material(name),
_conductivity(k),
_density(rho),
_capacity(0.0)
{
}




conductorMaterial :: conductorMaterial(const std::string& name) :
material(name),
_conductivity(0.0),
_density(0.0),
_capacity(0.0){

}




bool conductorMaterial :: check() const
{
	bool ret = true;

	if (_conductivity <= 0.0)
	{
		ret = false;
		std::cout << "Error in conductor material. Non-positive conductivity";
	}

    if (_capacity <= 0.0)
    {
        ret = false;
        std::cout << "Error in conductor material. Non-positive heat capacity";
    }

	return ret;
}




const double& conductorMaterial :: conductivity() const
{
    return _conductivity;
}




conductorMP* conductorMaterial :: createMaterialPoint() const
{
	conductorMP *mp = new conductorMP(*this);
	return mp;
}




double conductorMaterial :: density() const
{
    return _density;
}




const double& conductorMaterial :: heatCapacity() const
{
    return _capacity;
}




void conductorMaterial :: print(std::ostream &of) const
{
    of  << "\n   Isotropic thermally conducting material "
        << "\n   Conductivity : " << _conductivity
        << "\n   Density      : " << _density
        << "\n   Heat capacity: " << _capacity;
}




void conductorMaterial :: setRandom()
{
#ifdef STRICT_THREAD_SAFE
    theMutex.lock();
#endif

    _conductivity = muesli::randomUniform(1.0, 10.0);
    _density      = muesli::randomUniform(1.0, 10.0);
    _capacity     = muesli::randomUniform(1.0, 10.0);

#ifdef STRICT_THREAD_SAFE
    theMutex.unlock();
#endif
}




bool conductorMaterial :: test(std::ostream& os)
{
    bool isok = true;
    setRandom();

#ifdef STRICT_THREAD_SAFE
    theMutex.lock();
#endif

    conductorMP* p = this->createMaterialPoint();

    isok = p->testImplementation(os);
    delete p;

#ifdef STRICT_THREAD_SAFE
    theMutex.unlock();
#endif
    return isok;
}




double conductorMaterial :: getProperty(const propertyName p) const
{
	double ret=0.0;

    if      (p == PR_CONDUCTIVITY)		ret = _conductivity;
    else if (p == PR_THERMAL_CAP)       ret = _capacity;
    else
    {
      std:: cout << "\n Error in GetconductorMaterialProperty.";
    }
	return ret;
}




conductorMP :: conductorMP(const conductorMaterial& theMaterial)
: mat(&theMaterial),
time_n(0.0), time_c(0.0)
{
    gradT_n.setZero();
    gradT_c.setZero();
}




void  conductorMP :: commitCurrentState()
{
    time_c = time_n;
    gradT_n = gradT_c;
}




void conductorMP :: contractTangent(const ivector& na, const ivector& nb, double& tg) const
{
    tg = mat->_conductivity * na.dot(nb);
}




double conductorMP :: density() const
{
    return mat->density();
}




materialState conductorMP :: getConvergedState() const
{
    materialState state;

    state.theTime = time_n;
    state.theVector.push_back(gradT_n);

    return state;
}




materialState conductorMP :: getCurrentState() const
{
    materialState state;

    state.theTime = time_c;
    state.theVector.push_back(gradT_c);

    return state;
}




void conductorMP :: heatflux(ivector &q) const
{
	q = (- mat->_conductivity) * gradT_c;
}




void conductorMP :: resetCurrentState()
{
#ifdef STRICT_THREAD_SAFE
    theMutex.lock();
#endif

    time_c  = time_n;
    gradT_c = gradT_n;

#ifdef STRICT_THREAD_SAFE
    theMutex.unlock();
#endif
}




void conductorMP :: setRandom()
{
#ifdef STRICT_THREAD_SAFE
    theMutex.lock();
#endif

    gradT_n.setRandom();
    gradT_c = gradT_n;

#ifdef STRICT_THREAD_SAFE
    theMutex.unlock();
#endif
}




double conductorMP :: thermalEnergy() const
{
 	return 0.5*mat->_conductivity * gradT_c.dot(gradT_c);
}





bool conductorMP :: testImplementation(std::ostream& os) const
{
    bool isok = true;


    // compare heat flux with derivative of energy
    if (true)
    {
      double tn1 = muesli::randomUniform(0.1,1.0);

        // programmed heatflux
        ivector gradT;
        gradT.setRandom();
        const_cast<conductorMP*>(this)->updateCurrentState(tn1, gradT);

        ivector q;
        heatflux(q);
        q *= -1.0;


        // numerical differentiation heatflux
        ivector numFlux;
        numFlux.setZero();
        double   inc = 1.0e-3;

        for (size_t i=0; i<3; i++)
        {
            double original = gradT(i);

            gradT(i) = original + inc;
            const_cast<conductorMP*>(this)->updateCurrentState(tn1, gradT);
            double Wp1 = thermalEnergy();

            gradT(i) = original + 2.0*inc;
            const_cast<conductorMP*>(this)->updateCurrentState(tn1, gradT);
            double Wp2 = thermalEnergy();

            gradT(i) = original - inc;
            const_cast<conductorMP*>(this)->updateCurrentState(tn1, gradT);
            double Wm1 = thermalEnergy();

            gradT(i) = original - 2.0*inc;
            const_cast<conductorMP*>(this)->updateCurrentState(tn1, gradT);
            double Wm2 = thermalEnergy();


            // fourth order approximation of the derivative
            double der = (-Wp2 + 8.0*Wp1 - 8.0*Wm1 + Wm2)/(12.0*inc);
            numFlux(i) = der;

            gradT(i) = original;
        }

        // relative error less than 0.01%
        ivector error = numFlux - q;
        isok = (error.norm()/q.norm() < 1e-4);

        os << "\n   1. Comparing heat flux with derivative of thermal energy.";

        if (!isok)
        {
            os << "\n Relative error in q computation %e. Test failed." <<  error.norm()/q.norm();
            os << "\n   Heat flux:\n" << q;
            os << "\n   Numeric heat flux:\n" << numFlux;
        }
        else
            os << " Test passed.";
    }

    return isok;
}




void conductorMP :: updateCurrentState(const double theTime, const ivector& gradT)
{
#ifdef STRICT_THREAD_SAFE
    theMutex.lock();
#endif

    time_c  = theTime;
    gradT_c = gradT;

#ifdef STRICT_THREAD_SAFE
    theMutex.unlock();
#endif
}

