/****************************************************************************
 *
 *                                 M U E S L I   v 1.4
 *
 *
 *     Copyright 2016 IMDEA Materials Institute, Getafe, Madrid, Spain
 *     Contact: muesli.materials@imdea.org
 *     Author: Ignacio Romero (ignacio.romero@imdea.org)
 *
 *     This file is part of MUESLI.
 *
 *     MUESLI is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     MUESLI is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with MUESLI.  If not, see <http://www.gnu.org/licenses/>.
 *
*****************************************************************************/



#include "muesli/material.h"
#include <string>
#include <map>
#include <vector>
#include <cmath>
#include <cstdlib>
#include <cstdarg>
#include <cstdio>
#include <cstring>
#include <sstream>
#include <stdexcept>

using namespace muesli;


std::ostream* material::materialLog(0);


material :: material() :
_name("unnamed material"),
_label(0)
{
    if (materialLog == 0) materialLog = &(std::cout);
}




material :: material(const std::string& name) :
_name(name),
_label(0)
{
    if (materialLog == 0) materialLog = &(std::cout);
}




material :: material(const std::string& name,
                     const materialProperties& cl)
:
_name(name),
_label(0)
{
    if (materialLog == 0) materialLog = &(std::cout);

    double dlabel;
    muesli::assignValue(cl, "label", dlabel);
    _label = static_cast<size_t>(dlabel);
}




material :: ~material()
{}




std::ostream& material :: material::getLogger()
{
    if (materialLog == 0) materialLog = &(std::cout);
    return *materialLog;
}




bool material :: check() const
{
    bool isok = true;

    if (_label == 0) isok = false;
    if (_name.empty()) isok = false;

    return isok;
}




void material :: setLogger(std::ostream &of)
{
    material::materialLog = &of;
}




materialPoint :: materialPoint()
{}




materialPoint :: materialPoint(material &m)
{
}




void  materialDB :: addMaterial(const size_t label, material& m)
{
#ifdef STRICT_THREAD_SAFE
    theMutex.lock();
#endif
    theDB[label] = &m;
#ifdef STRICT_THREAD_SAFE
    theMutex.unlock();
#endif
}




void materialDB::free()
{
#ifdef STRICT_THREAD_SAFE
    theMutex.lock();
#endif

    std::map<size_t, muesli::material*>::iterator it = theDB.begin();
    while (it != theDB.end())
    {
        delete (*it).second;
        ++it;
    }

#ifdef STRICT_THREAD_SAFE
    theMutex.unlock();
#endif
}




materialDB& materialDB::getDB()
{
    static materialDB instance;
    return instance;
}




/* retrieves a material structure given its number in the static list of materials */
material& materialDB :: getMaterial(const size_t label) const
{
    std::map<size_t,material*>::const_iterator iter = theDB.find(label);

    if (iter == theDB.end())
    {
        std::ostream& mlog = material::getLogger();
        mlog << "Material with label " << label << " not defined";
    }
    return *(iter->second);
}




material& materialDB :: getMaterial(const std::string& name) const
{
    material *s(0);
    std::map<size_t,material*>::const_iterator iter = theDB.begin();

    while( iter != theDB.end() )
    {
        if ( iter->second->name() == name)
        {
            s = iter->second;
            break;
        }
        ++iter;
    }

    if (s == 0) throw std::runtime_error("Material not found in global list.");
    return *s;
}




void materialDB :: print(std::ostream &of) const
{
    std::ostream& mlog = material::getLogger();

    mlog << "\n\n\n\n\n";
    std::stringstream title;
    of  << "         M a t e r i a l s  (" << theDB.size() << ")";

    std::map<size_t,material*>::const_iterator iter = theDB.begin();
    while( iter != theDB.end() )
    {
        mlog << "\n\nMaterial #" << iter->first << ": " << iter->second->name();
        (iter->second)->print(mlog);
        ++iter;
    }
}



namespace muesli
{


    bool assignValue(const materialProperties& cl,
                     const std::string& key,
                     double& v)
    {
        bool found=false;
        materialProperties::const_iterator iter = cl.find(key);

        if (iter != cl.end())
        {
            v = iter->second;
            found = true;
        }
        return found;
    }




    bool assignValue(const materialProperties& cl,
                     const std::string& key,
                     std::vector<double>& v)
    {
        std::pair <materialProperties::const_iterator, materialProperties::const_iterator> foundObjects = cl.equal_range(key);

        materialProperties::const_iterator iter = foundObjects.first;
        while (iter != foundObjects.second)
        {
            v.push_back( iter->second );
            ++iter;
        }

        return !v.empty();
    }




    // keywords with a string option are stored in a strange way: keyword[space]option
    // since keywords can not contain spaces, the [space] is a unique separator
    bool assignValue(const materialProperties& cl,
                     const std::string& key,
                     std::string& v)
    {
        bool found=false;
        materialProperties::const_iterator iter = cl.begin();

        while ( iter != cl.end())
        {
            if ( iter->first.compare(0, key.length(), key) == 0)
            {
                v = iter->first.substr( key.length()+1, iter->first.length());
                found = true;
                break;
            }
            ++iter;
        }


        return found;
    }




    double randomUniform(const double low, const double up)
    {
        static bool initialized = false;

        if (!initialized)
        {
            srand((unsigned)time(0));
            initialized = true;
        }

        int     random_int = rand();
        double  random_dou = ( static_cast<double_t>(random_int))/ static_cast<double_t>(RAND_MAX);

        return random_dou * (up-low) + low;
    }




    int  discreteUniform(const int low, const int up)
    {
        int r;
        if (up == low)
            r = low;
        else
            r = low + rand() % (up + 1 -low);
        return r;
    }

}

