/****************************************************************************
 *
 *                                 M U E S L I   v 1.4
 *
 *
 *     Copyright 2016 IMDEA Materials Institute, Getafe, Madrid, Spain
 *     Contact: muesli.materials@imdea.org
 *     Author: Ignacio Romero (ignacio.romero@imdea.org)
 *
 *     This file is part of MUESLI.
 *
 *     MUESLI is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     MUESLI is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with MUESLI.  If not, see <http://www.gnu.org/licenses/>.
 *
*****************************************************************************/



#pragma once
#ifndef _muesli_material_
#define _muesli_material_

#include <stdio.h>
#include <map>
#include <vector>
#include <iostream>
#include <string>
#include "muesli/tensor.h"

#ifdef STRICT_THREAD_SAFE
#include <mutex>
#endif

// this is a convenient way to store all the possible material properties,
// and speed up their lookup
enum propertyName
{
    PR_LAMBDA, PR_MU,	// Lame coefficients
    PR_YOUNG,           // Young's modulus
    PR_POISSON,         // Poisson's ratio
    PR_BULK,            // Bulk modulus
    PR_CP,              // P waves velocity
    PR_CS,              // S waves velocity
    PR_SMAX,            // maximum normal stress
    PR_GF,              // fracture energy
    PR_PLSTRESS_C,      // plane stress wave velocity
    PR_ALPHA,           // thermal conductivity
    PR_TREF,            // reference temperature
    PR_CONDUCTIVITY,    // thermal conductivity
    PR_HEAT_SUPPLY,     // heat supply per unit volume
    PR_THERMAL_CAP,     // thermal capacity
    PR_THERMAL_EXP,     // thermal expansion coeff
    PR_HISO,            // isotropic hardening
    PR_HKINE,           // kinematic hardening
    PR_NU,              // dynamic viscosity
    PR_YIELD,           // yield stress
    PR_ISOHARD,         // isotropic hardening
    PR_KINHARD,         // kinematic hardening
    PR_ETA,
    PR_SRREF,
    PR_VEXPONENT
};



namespace muesli
{
    typedef std::multimap<std::string,double> materialProperties;

    class materialPoint;

    class material
    {

    public:
                                    material();
                                    material(const std::string& name);
                                    material(const std::string& name,
                                             const materialProperties& cl);
        virtual                     ~material();

        // operations
        virtual bool                check() const;
        virtual double              density() const=0;
        virtual double              getProperty(const propertyName p) const=0;
        virtual void                print(std::ostream &of=std::cout) const=0;
        virtual void                setRandom()=0;


        // this is the log file for all material data
        static void                 setLogger(std::ostream &of);
        static std::ostream&        getLogger();


        // information
        const size_t&               label() const;
        const std::string&          name() const;


    private:
        static std::ostream*        materialLog;
        std::string                 _name;
        size_t                      _label;
    };

    // inlines
    inline const size_t&            material::label() const {return _label;}
    inline const std::string&       material::name() const {return _name;}



    class materialState
    {
    public:
        double                      theTime;
        std::vector<double>         theDouble;
        std::vector<ivector>        theVector;
        std::vector<istensor>       theStensor;
        std::vector<itensor>        theTensor;
    };



    class materialPoint
    {

    public:
                                    materialPoint();
                                    materialPoint(material &m);
        virtual                     ~materialPoint(){};

        virtual materialState       getConvergedState() const = 0;
        virtual materialState       getCurrentState() const = 0;

        virtual void                commitCurrentState()=0;
        virtual void                resetCurrentState()=0;
    };




    // the materialDB is a convenient way to keep all the materials
    // in a single db which has convenient access
    class materialDB
    {

    public:

        // recover singleton
        static materialDB&          getDB();

        void                        addMaterial(const size_t label, muesli::material& m);
        muesli::material&           getMaterial(const size_t label) const;
        muesli::material&           getMaterial(const std::string& name) const;
        void                        free();
        void                        print(std::ostream &of=std::cout) const;


    private:

        std::map<size_t, muesli::material*> theDB;


                                    materialDB(){}
                                    materialDB(materialDB const& copy);
        materialDB&                 operator=(materialDB const& copy);
#ifdef STRICT_THREAD_SAFE
        std::mutex                  theMutex;
#endif
    };


    // utility functions
    double                          randomUniform(const double lower, const double upper);
    int                             discreteUniform(const int low, const int up);
    bool                            assignValue(const materialProperties& cl,
                                                const std::string& key,
                                                double& v);
    bool                            assignValue(const materialProperties& cl,
                                                const std::string& key,
                                                std::vector<double>& v);
    bool                            assignValue(const materialProperties& cl,
                                                const std::string& key,
                                                std::string& v);
}


#endif
